import java.text.*;
import java.util.*;

/**
 *
 * Instances of the <code>AssemblyInstruction</code> class represent 
 * assembly instructions of the CPU. The class contains methods to convert
 * instructions between their equivalent mnemonic forms and code forms, and
 * to check their validity.
 *
 * @author   Benedicto J. Catalan Jr.
 * @version  1.0, 02/13/2000
 * @since    CPU Simulator 1.1
 */
public
class AssemblyInstruction
{
   // String constants for each assembly instruction
   public static final String NOP = new String( "NOP" );
   public static final String LDAC = new String( "LDAC" );
   public static final String STAC = new String( "STAC" );
   public static final String MVAC = new String( "MVAC" );
   public static final String MOVR = new String( "MOVR" );
   public static final String JUMP = new String( "JUMP" );
   public static final String JMPZ = new String( "JMPZ" );
   public static final String JPNZ = new String( "JPNZ" );
   public static final String ADD = new String( "ADD" );
   public static final String SUB = new String( "SUB" );
   public static final String INAC = new String( "INAC" );
   public static final String CLAC = new String( "CLAC" );
   public static final String AND = new String( "AND" );
   public static final String OR = new String( "OR" );
   public static final String XOR = new String( "XOR" );
   public static final String NOT = new String( "NOT" );
   public static final String END = new String( "END" );

   public static final short opcodeNOP = 0x00;
   public static final short opcodeLDAC = 0x01;
   public static final short opcodeSTAC = 0x02;
   public static final short opcodeMVAC = 0x03;
   public static final short opcodeMOVR = 0x04;
   public static final short opcodeJUMP = 0x05;
   public static final short opcodeJMPZ = 0x06;
   public static final short opcodeJPNZ = 0x07;
   public static final short opcodeADD = 0x08;
   public static final short opcodeSUB = 0x09;
   public static final short opcodeINAC = 0x0a;
   public static final short opcodeCLAC = 0x0b;
   public static final short opcodeAND = 0x0c;
   public static final short opcodeOR = 0x0d;
   public static final short opcodeXOR = 0x0e;
   public static final short opcodeNOT = 0x0f;
   public static final short opcodeEND = 0xff;

   /**
    * Converts a string to its equivalent code byte.
    *
    * @param      mnemonic  the assembly mnemonic to be converted
    * @return     the code byte as a short, or <code>-1</code> if the string
    *                is not an assembly mnemonic.
    * @since      CPU Simulator 1.1
    */
   public static short toMnemonicCode( String mnemonic )
   {

      if ( NOP.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeNOP );
      }
      else if ( LDAC.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeLDAC );
      }
      else if ( STAC.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeSTAC );
      }
      else if ( MVAC.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeMVAC );
      }
      else if ( MOVR.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeMOVR );
      }
      else if ( JUMP.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeJUMP );
      }
      else if ( JMPZ.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeJMPZ );
      }
      else if ( JPNZ.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeJPNZ );
      }
      else if ( ADD.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeADD );
      }
      else if ( SUB.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeSUB );
      }
      else if ( INAC.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeINAC );
      }
      else if ( CLAC.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeCLAC );
      }
      else if ( AND.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeAND );
      }
      else if ( OR.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeOR );
      }
      else if ( XOR.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeXOR );
      }
      else if ( NOT.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeNOT );
      }
      else if ( END.equalsIgnoreCase( mnemonic ) )
      {
         return ( opcodeEND );
      }
      else
      {
         // Return -1 if failed to match assembly mnemonic
         return ( -1 );
      }

   }

   /**
    * Converts a code byte to its equivalent assembly mnemonic.
    *
    * @param      mnemonic  the code byte to be converted as a short
    * @return     the assembly mnemonic, or <code>null</code> if the code
    *                byte does not represent an assembly mnemonic.
    * @since      CPU Simulator 1.1
    */
   public static String toMnemonic( short code )
   {

      switch ( code )
      {

         case opcodeNOP:
            return ( NOP );

         case opcodeLDAC:
            return ( LDAC );

         case opcodeSTAC:
            return ( STAC );

         case opcodeMVAC:
            return ( MVAC );

         case opcodeMOVR:
            return ( MOVR );

         case opcodeJUMP:
            return ( JUMP );

         case opcodeJMPZ:
            return ( JMPZ );

         case opcodeJPNZ:
            return ( JPNZ );

         case opcodeADD:
            return ( ADD );

         case opcodeSUB:
            return ( SUB );

         case opcodeINAC:
            return ( INAC );

         case opcodeCLAC:
            return ( CLAC );

         case opcodeAND:
            return ( AND );

         case opcodeOR:
            return ( OR );

         case opcodeXOR:
            return ( XOR );

         case opcodeNOT:
            return ( NOT );

         case opcodeEND:
            return ( END );

         default:
            return ( null );

      }

   }

   public static short[] toInstructionCode( String[] instruction )
   {

      if ( ( instruction == null ) || ( instruction.length == 0 ) )
      {
         return ( null );
      }
      else
      {

         if ( ( instruction[ 0 ] != null ) && ( instruction[ 0 ].
            equalsIgnoreCase( NOP ) || instruction[ 0 ].equalsIgnoreCase(
            MVAC ) || instruction[ 0 ].equalsIgnoreCase( MOVR ) ||
            instruction[ 0 ].equalsIgnoreCase( ADD ) || instruction[ 0 ].
            equalsIgnoreCase( SUB ) || instruction[ 0 ].equalsIgnoreCase(
            INAC ) || instruction[ 0 ].equalsIgnoreCase( CLAC ) ||
            instruction[ 0 ].equalsIgnoreCase( AND ) || instruction[ 0 ].
            equalsIgnoreCase( OR ) || instruction[ 0 ].equalsIgnoreCase(
            XOR ) || instruction[ 0 ].equalsIgnoreCase( NOT ) || 
            instruction[ 0 ].equalsIgnoreCase( END )) && ( instruction.
            length < 2 ) )
         {
            short[] instructionCode = new short[ 1 ];

            instructionCode[ 0 ] = toMnemonicCode( instruction[ 0 ] );

            return ( instructionCode );
         }
         else if ( ( instruction[ 0 ] != null ) && ( instruction[ 0 ].
            equalsIgnoreCase( LDAC ) || instruction[ 0 ].equalsIgnoreCase(
            STAC ) || instruction[ 0 ].equalsIgnoreCase( JUMP ) ||
            instruction[ 0 ].equalsIgnoreCase( JMPZ ) || instruction[ 0 ].
            equalsIgnoreCase( JPNZ ) ) && ( instruction.length < 3 ) &&
            isAddress( instruction[ 1 ] ) )
         {
            short[] instructionCode = new short[ 3 ];
            short[] addressCode = toAddressCode( toAddressInteger(
               instruction[ 1 ] ) );

            instructionCode[ 0 ] = toMnemonicCode( instruction[ 0 ] );
            instructionCode[ 1 ] = addressCode[ 0 ];
            instructionCode[ 2 ] = addressCode[ 1 ];

            return ( instructionCode );
         }

         return ( null );
      }

   }

   /**
    * Converts a string to the equivalent numeric address as an integer.
    *
    * @param      addressString  string representation of an assembly
    *                address
    * @return     the address as an integer, or <code>-1</code> if the string
    *                does not represent a valid address
    * @exception  NumberFormatException  if the string does not represent a
    *                valid numeric address.
    * @since      CPU Simulator 1.1
    */
   public static int toAddressInteger( String addressString )
   {
      int address;

      if ( ( addressString != null ) && ( ! addressString.equals( "" ) ) )
      {
         char baseSpecifier = addressString.charAt( addressString.length() -
            1 );
         int addressRadix = toRadix( baseSpecifier );

         if ( addressRadix != -1 )
         {
            addressString = addressString.substring( 0, addressString.
               length() - 1 );
         }
         else
         {
            addressRadix = 10;
         }

         try
         {
            address = Integer.valueOf( addressString, addressRadix ).
               intValue();
         }
         catch ( NumberFormatException e )
         {
            return ( -1 );
         }

      }
      else
      {
         return ( -1 );
      }

      if ( isAddress( address ) )
      {
         return ( address );
      }
      else
      {
         return ( -1 );
      }

   }

   /**
    * Converts an integer to the equivalent code bytes.
    *
    * @param      address  the assembly address
    * @return     array of short containing the equivalent code bytes, or
    *                <code>null</code> if the integer is not a valid address
    * @since      CPU Simulator 1.1
    */
   public static short[] toAddressCode( int address )
   {

      if ( isAddress( address ) )
      {
         short[] codeByteArray = { ( short ) ( address % 0x100 ), ( short )
            ( ( address / 0x100 ) % 0x100 ) };

         return ( codeByteArray );
      }
      else
      {
         return ( null );
      }

   }

   /**
    * Returns radix corresponding to the base specifier character.
    *
    * @param      ch  the base specifier character.
    * @return     radix as an integer, or <code>-1</code> if character is not
    *                a valid base specifier.
    * @since      CPU Simulator 1.1
    */
   public static int toRadix( char ch )
   {
      ch = Character.toLowerCase( ch );

      if ( ch == 'b' )
      {
         return ( 2 );
      }
      else if ( ( ch == 'o' ) || ( ch == 'q' ) )
      {
         return ( 8 );
      }
      else if ( ch == 'd' )
      {
         return ( 10 );
      }
      else if ( ch == 'h' )
      {
         return ( 16 );
      }
      else
      {
         return ( -1 );
      }

   }

   public static boolean isConstantByte( short num )
   {
      return ( ( num >= 0 ) && ( num <= 0xFF ) );
   }

   public static boolean isConstantWord( int num )
   {
      return ( ( num >= 0 ) && ( num <= 0xFFFF ) );
   }

   /**
    * Determines whether or not the specified string is a valid assembly
    * mnemonic.
    *
    * @param      str  the string to be checked.
    * @return     <code>true</code> if the string is a valid assembly
    *                mnemonic; <code>false</code> otherwise.
    * @since      CPU Simulator 1.1
    */
   public static boolean isMnemonic( String str )
   {

      if ( NOP.equalsIgnoreCase( str ) ||
           LDAC.equalsIgnoreCase( str ) ||
           STAC.equalsIgnoreCase( str ) ||
           MVAC.equalsIgnoreCase( str ) ||
           MOVR.equalsIgnoreCase( str ) ||
           JUMP.equalsIgnoreCase( str ) ||
           JMPZ.equalsIgnoreCase( str ) ||
           JPNZ.equalsIgnoreCase( str ) ||
           ADD.equalsIgnoreCase( str ) ||
           SUB.equalsIgnoreCase( str ) ||
           INAC.equalsIgnoreCase( str ) ||
           CLAC.equalsIgnoreCase( str ) ||
           AND.equalsIgnoreCase( str ) ||
           OR.equalsIgnoreCase( str ) ||
           XOR.equalsIgnoreCase( str ) ||
           NOT.equalsIgnoreCase( str ) ||
           END.equalsIgnoreCase( str )    )
      {
         return true;
      }

      return false;
   }

   /**
    * Determines whether or not the specified integer is a valid assembly
    * address.
    *
    * @param      num  the integer to be checked.
    * @return     <code>true</code> if the integer is a valid assembly
    *                address; <code>false</code> otherwise.
    * @since      CPU Simulator 1.1
    */
   public static boolean isAddress( int num )
   {
      return ( ( num >= 0 ) && ( num <= 0xFFFF ) );
   }

   public static boolean isByte( short num )
   {
      return ( ( num >= 0 ) && ( num <= 0xFF ) );
   }

   public static boolean isWord( int num )
   {
      return ( ( num >= 0 ) && ( num <= 0xFFFF ) );
   }

   /**
    * Determines whether or not the specified string is a valid assembly
    * address.
    *
    * @param      str  the string to be checked.
    * @return     <code>true</code> if the string is a valid assembly
    *                address; <code>false</code> otherwise.
    * @since      CPU Simulator 1.1
    */
   public static boolean isAddress( String str )
   {
      return ( toAddressInteger( str ) != -1 );
   }

   /**
    * Determines whether or not the specified strings are valid assembly
    * operands.
    *
    * @param      str  the string to be checked.
    * @return     <code>true</code> if the string is a valid assembly
    *                operand; <code>false</code> otherwise.
    * @since      CPU Simulator 1.1
    */
   public static boolean isValidOperands( String mnemonic, String[]
      operands )
   {

      if ( mnemonic.equalsIgnoreCase( NOP ) || mnemonic.equalsIgnoreCase(
         MVAC ) || mnemonic.equalsIgnoreCase( MOVR ) || mnemonic.
         equalsIgnoreCase( ADD ) || mnemonic.equalsIgnoreCase( SUB ) ||
         mnemonic.equalsIgnoreCase( INAC ) || mnemonic.equalsIgnoreCase(
         CLAC ) || mnemonic.equalsIgnoreCase( AND ) || mnemonic.
         equalsIgnoreCase( OR ) || mnemonic.equalsIgnoreCase( XOR ) ||
         mnemonic.equalsIgnoreCase( NOT ) || mnemonic.equalsIgnoreCase( 
         END ) )
      {
         return ( ( operands == null ) || ( operands.length == 0 ) || ( (
            operands.length == 1 ) && ( operands[ 0 ].equals( "" ) ) ) );
      }
      else if ( mnemonic.equalsIgnoreCase( LDAC ) || mnemonic.
         equalsIgnoreCase( STAC ) || mnemonic.equalsIgnoreCase( JUMP ) ||
         mnemonic.equalsIgnoreCase( JMPZ ) || mnemonic.equalsIgnoreCase(
         JPNZ ) )
      {
         return ( ( operands != null ) && ( operands.length == 1 ) &&
            isAddress( operands[ 0 ] ) );
      }

      return ( false );
   }

   public static int expectsOperands( String mnemonic )
   {

      if ( mnemonic.equalsIgnoreCase( NOP ) || mnemonic.equalsIgnoreCase(
         MVAC ) || mnemonic.equalsIgnoreCase( MOVR ) || mnemonic.
         equalsIgnoreCase( ADD ) || mnemonic.equalsIgnoreCase( SUB ) ||
         mnemonic.equalsIgnoreCase( INAC ) || mnemonic.equalsIgnoreCase(
         CLAC ) || mnemonic.equalsIgnoreCase( AND ) || mnemonic.
         equalsIgnoreCase( OR ) || mnemonic.equalsIgnoreCase( XOR ) ||
         mnemonic.equalsIgnoreCase( NOT ) || mnemonic.equalsIgnoreCase( 
         END ) )
      {
         return ( 0 );
      }
      else if ( mnemonic.equalsIgnoreCase( LDAC ) || mnemonic.
         equalsIgnoreCase( STAC ) || mnemonic.equalsIgnoreCase( JUMP ) ||
         mnemonic.equalsIgnoreCase( JMPZ ) || mnemonic.equalsIgnoreCase(
         JPNZ ) )
      {
         return ( 1 );
      }

      return ( -1 );
   }

   public static short toByteShort( String byteString )
   {
      short byteShort;

      if ( ( byteString != null ) && ( ! byteString.equals( "" ) ) )
      {
         char baseSpecifier = byteString.charAt( byteString.length() - 1 );
         int byteRadix = toRadix( baseSpecifier );

         if ( byteRadix != -1 )
         {
            byteString = byteString.substring( 0, byteString.length() - 
               1 );
         }
         else
         {
            byteRadix = 10;
         }

         try
         {
            byteShort = Short.valueOf( byteString, byteRadix ).
               shortValue();
         }
         catch ( NumberFormatException e )
         {
            return ( -1 );
         }

      }
      else
      {
         return ( -1 );
      }

      if ( isByte( byteShort ) )
      {
         return ( byteShort );
      }
      else
      {
         return ( -1 );
      }

   }

   public static int toWordInteger( String wordString )
   {
      int wordInteger;

      if ( ( wordString != null ) && ( ! wordString.equals( "" ) ) )
      {
         char baseSpecifier = wordString.charAt( wordString.length() - 1 );
         int wordRadix = toRadix( baseSpecifier );

         if ( wordRadix != -1 )
         {
            wordString = wordString.substring( 0, wordString.length() - 
               1 );
         }
         else
         {
            wordRadix = 10;
         }

         try
         {
            wordInteger = Integer.valueOf( wordString, wordRadix ).
               intValue();
         }
         catch ( NumberFormatException e )
         {
            return ( -1 );
         }

      }
      else
      {
         return ( -1 );
      }

      if ( isWord( wordInteger ) )
      {
         return ( wordInteger );
      }
      else
      {
         return ( -1 );
      }

   }

   public static short[] toWordCode( int word )
   {

      if ( isWord( word ) )
      {
         short[] codeByteArray = { ( short ) ( word % 0x100 ), ( short )
            ( ( word / 0x100 ) % 0x100 ) };

         return ( codeByteArray );
      }
      else
      {
         return ( null );
      }

   }

   public static String toNumberString( long num, int radix, int length )
   {
      String str = Long.toString( num, radix );

      while ( str.length() < length )
      {
         str = "0" + str;
      }

      return ( str.substring( str.length() - length, str.length() ) );
   }

}
